/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2026 i-net software GmbH, Berlin, Germany.
**/
package cache;

import java.io.File;//File, FielOutPutStream
import java.io.FileOutputStream;
import java.util.Properties;

import com.inet.report.Engine;//Engine, RDC
import com.inet.report.RDC;
import com.inet.report.ReportException;
import com.inet.report.cache.Cache;//Cache,ReportCacheKey
import com.inet.report.cache.ReportCacheKey;

/**
 * This sample will do the following steps: - Save the report file in the i-net Clear Reports report file format (only
 * necessary if the rpt file is not an i-net Clear Reports report file). - Store the rendered report into the Cache.
 * Application flow: 1. create an engine object from .rpt file (e.g. a Crystal Reports report file) 2. save engine
 * object as xml file (save as i-net Clear Reports report file) 3. create engine object from .xml file (load i-net Clear
 * Reports file and execute it) 4. put the rendered report in the cache 5. save the rendered report into a file by
 * receiving the data from Cache (Export)
 */
public class CacheLoadEngine {

    private static String reportPath   = "samples/sample.rpt";            // Set here the path of your report
    private static String outputFormat = Engine.EXPORT_PDF;               // Set here the export format
    private static String outputPath   = "samples/sample." + outputFormat; // Set here the path of the output file

    /**
     * Main method of the sample.
     * @param args no parameters used
     * @throws Exception if an error occurred during execution of the sample
     */
    public static void main( String[] args ) throws Exception {
        //serialize the engine, only if the rpt file is not in the XML report file format.
        engineToXML();
        //load Engine from xml file, set Export format to pdf. See also RDC.loadEngine(File f).
        Engine eng = RDC.loadEngine( new File( reportPath ), outputFormat );

        Cache cache = Cache.getCache();
        //create Properties for needed ReportCacheKey
        Properties engineProperties = new Properties();
        engineProperties.put( "report", reportPath );
        /* you can put additional properties, that goes into the key
         * e.g.:
         * engineProperties.put("export_fmt",outputFormat);
         * engineProperties.put("dateOfCreation","11/11/2003");
         */
        ReportCacheKey key = cache.createKey( engineProperties );
        /*
        *if you want to put the latest report in cache, use the follwing code: 
        *--------------------------------------------------------------------------
        *if(cache.exists(key)){
        * 	cache.delete(serializedEngine);//needs the value from "report" property
        *   eng.execute();//running the report
        *   cache.addEngine(eng, key);
        * }
        *else 
        * 	cache.addEngine(eng, key);
        *--------------------------------------------------------------------------
        *
        *if you want to put the latest report in cache only when report is not in cache, use following code:
        *--------------------------------------------------------------------------
        */
        if( !cache.exists( key ) ) {
            eng.execute();//running the report
            cache.addEngine( eng, key );
        }
        //--------------------------------------------------------------------------

        //save the rendered report in a file to eye them later 
        byte[] totalReport = getTotalReport( cache, key );
        exportToFile( totalReport );

        // Stop Listener and quit program
        System.exit( 0 );
    }

    /**
     * Returns the data of an report from Cache
     * @param cache instance of the cache
     * @param key unique key for a report
     * @return byte array containing the data of an report
     * @throws ReportException if any error occurred
     */
    static final byte[] getTotalReport( Cache cache, ReportCacheKey key ) throws ReportException {
        //get the number of pages
        int pageCount = cache.getPageCountAndWait( key );
        //get the size of report
        int totalReportSize = cache.getTotalPageSizeAndWait( key );
        //create the ByteArray that stores the report
        byte[] reportData = new byte[totalReportSize];
        int offset = 0;
        //getting the data of each page
        for( int page = 1; page <= pageCount; page++ ) {
            byte[] pageData = cache.getPageAndWait( key, page );
            //put bytes in reportData
            System.arraycopy( pageData, 0, reportData, offset, pageData.length );
            offset += pageData.length;
        }
        return reportData;
    }

    /**
     * Method need to be use only if the rpt file is not an i-net Clear Reports report file.
     * @throws Exception if any error occurred
     */
    static final void engineToXML() throws Exception {
        Engine eng = new Engine( Engine.NO_EXPORT );
        eng.setReportFile( reportPath );
        RDC.saveEngine( new File( reportPath ), eng );//export format won't be saved in xml file
    }

    /**
     * Saves the result into a file.
     * @param reportdata byte array containing the data of an report
     * @throws Exception if any error occurred
     */
    static final void exportToFile( byte[] reportdata ) throws Exception {
        FileOutputStream fos = new FileOutputStream( outputPath );
        fos.write( reportdata );
        fos.flush();
        fos.close();
    }
}
